/**
 * OWASP AppSensor
 * 
 * This file is part of the Open Web Application Security Project (OWASP)
 * AppSensor project. For details, please see
 * <a href="http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project">
 * 	http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project</a>.
 *
 * Copyright (c) 2010 - The OWASP Foundation
 * 
 * AppSensor is published by OWASP under the BSD license. You should read and accept the
 * LICENSE before you use, modify, and/or redistribute this software.
 * 
 * @author Michael Coates <a href="http://www.aspectsecurity.com">Aspect Security</a>
 * @author John Melton <a href="http://www.jtmelton.com/">jtmelton</a>
 * @created 2010
 */
namespace org.owasp.appsensor
{

    using System;
    using System.Configuration;
    using System.Reflection;
    using System.Security;
    using Owasp.Esapi.Errors;

    /**
     * This class was cloned from Esapi for use in our code.  The previous versions used
     * the ObjFactory class directly from Esapi which was fine, but tied us to specific versions 
     * of Esapi as dependencies more closely since they kept modifying the API as part of the 2.0 const push.
     * This class was retrieved on 9/14/2010 and came from the 2.0rc7 release of Esapi. 
     * 
     * @author John Melton (jtmelton .at. gmail.com)
     *         <a href="http://www.jtmelton.com/">jtmelton</a>
     * @since Sept 14, 2010
     *
     */

    /**
     * A generic object factory to create an object of class T. T must be a concrete
     * class that has a no-argument public constructor or a implementor of the Singleton pattern
     * that has a no-arg static getInstance method. If the class being created has a getInstance
     * method, it will be used as a singleton and newInstance() will never be called on the
     * class no matter how many times it comes through this factory.
     *
     * <p>
     * Typical use is something like:
     * <pre>
     * 		import com.example.interfaces.DrinkingEstablishment;
     * 		import com.example.interfaces.Beer;
     * 		...
     * 		// Typically these would be populated from some Java properties file
     * 		String barName = "com.example.foo.Bar";
     * 		String beerBrand = "com.example.brewery.Guiness";
     * 		...
     * 		DrinkingEstablishment bar = ObjFactory.make(barName, "DrinkingEstablishment");
     * 		Beer beer = ObjFactory.make(beerBrand, "Beer");
     *		bar.drink(beer);	// Drink a Guiness beer at the foo Bar. :)
     *		...
     * </pre>
     * </p><p>
     *  Copyright (c) 2009 - The OWASP Foundation
     *  </p>
     * @author kevin.w.wall@gmail.com
     * @author Chris Schmidt ( chrisisbeef .at. gmail.com )
     */
    public class ASObjFactory
    {

        /**
         * Create an object based on the <code>className</code> parameter.
         * 
         * @param className	The name of the class to construct. Should be a fully qualified name and
         * 					generally the same as type <code>T</code>
         * @param typeName	A type name used in error messages / exceptions.
         * @return	An object of type <code>className</code>, which is cast to type <code>T</code>.
         * @throws	ConfigurationException thrown if class name not found in class path, or does not
         * 			have a public, no-argument constructor, or is not a concrete class, or if it is
         * 			not a sub-type of <code>T</code> (or <code>T</code> itself). Usually this is
         * 			caused by a misconfiguration of the class names specified in the Esapi.properties
         * 			file. Also thrown if the CTOR of the specified <code>className</code> throws
         * 			an <code>Exception</code> of some type.
         */
        public static T Make<T>(String className, String typeName)
        {
            object obj = null;
            String errMsg = null;
            try
            {
                if (String.IsNullOrEmpty(className))
                {
                    throw new ArgumentNullException("Classname cannot be null or empty.");
                }
                if (String.IsNullOrEmpty(typeName))
                {
                    // No big deal...just use "[unknown?]" for this as it's only for an err msg.
                    typeName = "[unknown?]";	// CHECKME: Any better suggestions?
                }

                Type theType = Assembly.GetExecutingAssembly().GetType(className);
                if (theType == null)
                {
                    throw new ConfigurationErrorsException("Type " + typeName + " not found");
                }

                try
                {
                    MethodInfo singleton = theType.GetMethod("GetInstance");
                    if (singleton != null)
                    {
                        // Tis a singleton
                        var classInstance = Activator.CreateInstance(theType, null);

                        // If the implementation class Contains a getInstance method that is not static, this is an invalid
                        // object configuration and a ConfigurationException will be thrown.
                        if (!singleton.IsStatic)
                        {
                            throw new ConfigurationErrorsException("Class [" + className + "] Contains a non-static getInstance method.");
                        }

                        obj = singleton.Invoke(classInstance, null);
                    }
                    else
                    {
                        // Not a singleton so just invoke default constructor
                        obj = Activator.CreateInstance(theType);
                    }
                }
                catch (AmbiguousMatchException e)
                {
                    // This is not clever, it means there are multiple GetInstance methods.
                    throw new ConfigurationErrorsException("Class [" + className + "] Contains multiple getInstance method.");
                }
                catch (SecurityException e)
                {
                    // The class is meant to be singleton, however, the SecurityManager restricts us from calling the
                    // GetInstance method on the class, thus this is a configuration issue and a ConfigurationException
                    // is thrown
                    throw new ConfigurationErrorsException("The SecurityManager has restricted the object factory from getting a reference to the singleton implementation" +
                            "of the class [" + className + "]", e);
                }

                return (T)obj;
            }
            catch (ConfigurationErrorsException ex)
            {
                // Do nowt since the exceptions are thrown lower down
                throw ex;
            }
            catch (ArgumentNullException ex)
            {
                errMsg = ex.ToString() + " " + typeName + " type name cannot be null or empty.";
                throw new ConfigurationErrorsException(errMsg, ex);
            }
            catch (ArgumentException ex)
            {
                errMsg = ex.ToString() + " " + className + " class name is invalid.";
                throw new ConfigurationErrorsException(errMsg, ex);
            }
            catch (MissingMethodException ex)
            {
                errMsg = ex.ToString() + " " + typeName + " class (" + className + ") must have a public, no-arg constructor.";
                throw new ConfigurationErrorsException(errMsg, ex);
            }
            catch (MemberAccessException ex)
            {
                errMsg = ex.ToString() + " " + typeName + " class (" + className + ") must be concrete.";
                throw new ConfigurationErrorsException(errMsg, ex);
            }
            catch (InvalidCastException ex)
            {
                errMsg = ex.ToString() + " " + typeName + " class (" + className + ") must be a subtype of T in ObjFactory<T>";
                throw new ConfigurationErrorsException(errMsg, ex);
            }
            catch (Exception ex)
            {
                // Because we are using reflection, we want to catch any checked or unchecked Exceptions and
                // re-throw them in a way we can handle them. Because using reflection to construct the object,
                // we can't have the compiler notify us of uncaught exceptions. For example, JavaEncryptor()
                // CTOR can throw [well, now it can] an EncryptionException if something goes wrong. That case
                // is taken care of here.
                //
                // CHECKME: Should we first catch RuntimeExceptions so we just let unchecked Exceptions go through
                //		    unaltered???
                //
                errMsg = ex.ToString() + " " + typeName + " class (" + className + ") CTOR threw exception.";
                throw new ConfigurationErrorsException(errMsg, ex);
            }
            // DISCUSS: Should we also catch ExceptionInInitializerError here? See Google Issue #61 comments.
        }

        /**
         * Not instantiable
         */
        private ASObjFactory() { }
    }
}